//--------------------------------------------------------------------------
//
//  Software for MSP430 based e-meters.
//
//  THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR
//  REPRESENTATIONS, EITHER EXPRESS, IMPLIED OR STATUTORY, 
//  INCLUDING ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS 
//  FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY OR 
//  COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE. 
//  TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET 
//  POSSESSION, AND NON-INFRINGEMENT OF ANY THIRD PARTY 
//  INTELLECTUAL PROPERTY RIGHTS WITH REGARD TO THE PROGRAM OR 
//  YOUR USE OF THE PROGRAM.
//
//  IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL, 
//  CONSEQUENTIAL OR INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY 
//  THEORY OF LIABILITY AND WHETHER OR NOT TI HAS BEEN ADVISED 
//  OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY OUT 
//  OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM. 
//  EXCLUDED DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF 
//  REMOVAL OR REINSTALLATION, COMPUTER TIME, LABOR COSTS, LOSS 
//  OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS, OR LOSS OF 
//  USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S 
//  AGGREGATE LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF 
//  YOUR USE OF THE PROGRAM EXCEED FIVE HUNDRED DOLLARS 
//  (U.S.$500).
//
//  Unless otherwise stated, the Program written and copyrighted 
//  by Texas Instruments is distributed as "freeware".  You may, 
//  only under TI's copyright in the Program, use and modify the 
//  Program without any charge or restriction.  You may 
//  distribute to third parties, provided that you transfer a 
//  copy of this license to the third party and the third party 
//  agrees to these terms by its first use of the Program. You 
//  must reproduce the copyright notice and any other legend of 
//  ownership on each copy or partial copy, of the Program.
//
//  You acknowledge and agree that the Program contains 
//  copyrighted material, trade secrets and other TI proprietary 
//  information and is protected by copyright laws, 
//  international copyright treaties, and trade secret laws, as 
//  well as other intellectual property laws.  To protect TI's 
//  rights in the Program, you agree not to decompile, reverse 
//  engineer, disassemble or otherwise translate any object code 
//  versions of the Program to a human-readable form.  You agree 
//  that in no event will you alter, remove or destroy any 
//  copyright notice included in the Program.  TI reserves all 
//  rights not specifically granted under this license. Except 
//  as specifically provided herein, nothing in this agreement 
//  shall be construed as conferring by implication, estoppel, 
//  or otherwise, upon you, any license or other right under any 
//  TI patents, copyrights or trade secrets.
//
//  You may not use the Program in non-TI devices.
//
//  File: basic_display.c
//
//  Steve Underwood <steve-underwood@ti.com>
//  Texas Instruments Hong Kong Ltd.
//
//  $Id: emeter-basic-display.c,v 1.22 2008/05/21 08:08:15 a0754793 Exp $
//
/*! \file emeter-structs.h */
//
//--------------------------------------------------------------------------
//
//  MSP430 simple LCD display routines for e-meters
//
#include <stdint.h>
#include <stdlib.h>
#include <io.h>
#include <emeter-toolkit.h>

#include "emeter-structs.h"

#if defined(MULTI_RATE_SUPPORT)
#include "emeter-multirate.h"
#endif

#if defined(__MSP430__)  &&  defined(CUSTOM_LCD_SUPPORT)
#include <lcd-segments.h>
/* Define hex digits and the minus sign to match the allocation of segments we are using. */
const lcd_cell_t lcd_digit_table[] =
{
    CHAR_0,
    CHAR_1,
    CHAR_2,
    CHAR_3,
    CHAR_4,
    CHAR_5,
    CHAR_6,
    CHAR_7,
    CHAR_8,
    CHAR_9,
    CHAR_A,
    CHAR_B,
    CHAR_C,
    CHAR_D,
    CHAR_E,
    CHAR_F
};

const lcd_cell_t char_minus = CHAR_MINUS;

const int lcd_cells = LCD_CELLS;
const int lcd_pos_base = LCD_POS_BASE;
const int lcd_pos_step = LCD_POS_STEP;
#endif

#if defined(MULTI_RATE_SUPPORT)
uint8_t info_step;
uint8_t info_substep;
#endif

#if defined(__MSP430__)  &&  defined(BASIC_LCD_SUPPORT)
#include <lcd-segments.h>
/* Define hex digits and the minus sign to match the allocation of segments we are using. */
const lcd_cell_t lcd_digit_table[] =
{
    CHAR_0,
    CHAR_1,
    CHAR_2,
    CHAR_3,
    CHAR_4,
    CHAR_5,
    CHAR_6,
    CHAR_7,
    CHAR_8,
    CHAR_9,
    CHAR_A,
    CHAR_B,
    CHAR_C,
    CHAR_D,
    CHAR_E,
    CHAR_F
};

    #if defined(USE_STARBURST)
const lcd_cell_t lcd_alpha_table[] =
{
    CHAR_A,
    CHAR_B,
    CHAR_C,
    CHAR_D,
    CHAR_E,
    CHAR_F,
    CHAR_G,
    CHAR_H,
    CHAR_I,
    CHAR_J,
    CHAR_K,
    CHAR_L,
    CHAR_M,
    CHAR_N,
    CHAR_O,
    CHAR_P,
    CHAR_Q,
    CHAR_R,
    CHAR_S,
    CHAR_T,
    CHAR_U,
    CHAR_V,
    CHAR_W,
    CHAR_X,
    CHAR_Y,
    CHAR_Z
};
    #endif

    #if defined(ASCII_LCD)
static const lcd_cell_t lcd_ascii_table[] =
{
    CHAR_SPACE,
    CHAR_SPACE,         //exclamation
    CHAR_DOUBLEQUOTE,
    CHAR_SPACE,         //hash
    CHAR_DOLLAR,
    CHAR_SPACE,         //percent
    CHAR_SPACE,         //ampersand
    CHAR_QUOTE,
    CHAR_LEFTBRACKET,
    CHAR_RIGHTBRACKET,
    CHAR_ASTERISK,
    CHAR_PLUS,
    CHAR_SPACE,         //comma
    CHAR_MINUS,
    CHAR_SPACE,         //full stop
    CHAR_SLASH,
    CHAR_0,
    CHAR_1,
    CHAR_2,
    CHAR_3,
    CHAR_4,
    CHAR_5,
    CHAR_6,
    CHAR_7,
    CHAR_8,
    CHAR_9,
    CHAR_SPACE,         //colon
    CHAR_SPACE,         //semi-colon
    CHAR_LT,
    CHAR_EQUALS,
    CHAR_GT,
    CHAR_QUESTION,
    CHAR_SPACE,         //at sign
    CHAR_A,
    CHAR_B,
    CHAR_C,
    CHAR_D,
    CHAR_E,
    CHAR_F,
    CHAR_G,
    CHAR_H,
    CHAR_I,
    CHAR_J,
    CHAR_K,
    CHAR_L,
    CHAR_M,
    CHAR_N,
    CHAR_O,
    CHAR_P,
    CHAR_Q,
    CHAR_R,
    CHAR_S,
    CHAR_T,
    CHAR_U,
    CHAR_V,
    CHAR_W,
    CHAR_X,
    CHAR_Y,
    CHAR_Z,
    CHAR_LEFTBRACKET,
    CHAR_BACKSLASH,
    CHAR_RIGHTBRACKET,
    CHAR_CARET,
    CHAR_UNDERSCORE,
    CHAR_BACKQUOTE,
    CHAR_a,
    CHAR_b,
    CHAR_C,
    CHAR_d,
    CHAR_e,
    CHAR_f,
    CHAR_g,
    CHAR_h,
    CHAR_i,
    CHAR_j,
    CHAR_k,
    CHAR_l,
    CHAR_m,
    CHAR_n,
    CHAR_o,
    CHAR_p,
    CHAR_q,
    CHAR_r,
    CHAR_s,
    CHAR_t,
    CHAR_u,
    CHAR_v,
    CHAR_w,
    CHAR_x,
    CHAR_y,
    CHAR_z,
    CHAR_LEFTBRACKET,
    CHAR_VERTICALBAR,
    CHAR_RIGHTBRACKET,
    CHAR_SPACE,         //squiggle
    CHAR_SPACE          //delete
};

void lcd_text(char *s, int pos)
{
    int x;

    if (lcd_pos_step < 0)
        pos = -pos;
    if (abs(lcd_pos_step) > 1)
        pos <<= 1;
    pos = lcd_pos_base + pos;
    while (*s)
    {
        x = lcd_ascii_table[*s++ - 0x20];
        LCDMEM[pos] = x & 0xFF;
        pos += (lcd_pos_step >> 1);
        LCDMEM[pos] = x >> 8;
        pos += (lcd_pos_step >> 1);
    }
}
    #endif

const lcd_cell_t char_minus = CHAR_MINUS;

const int lcd_cells = LCD_CELLS;
const int lcd_pos_base = LCD_POS_BASE;
const int lcd_pos_step = LCD_POS_STEP;

static const lcd_cell_t lcd_high[TEXT_MESSAGE_LENGTH] =
{
    CHAR_H,
    CHAR_i,
    CHAR_g,
    CHAR_h,
    CHAR_SPACE,
    CHAR_SPACE,
    #if TEXT_MESSAGE_LENGTH == 7
    CHAR_SPACE
    #endif
};

static const lcd_cell_t lcd_startup[TEXT_MESSAGE_LENGTH] =
{
    #if defined(USE_STARBURST)
    CHAR_S,
    CHAR_T,
    CHAR_A,
    CHAR_R,
    CHAR_T,
    CHAR_SPACE,
    #else
    CHAR_S,
    CHAR_t,
    CHAR_a,
    CHAR_r,
    CHAR_t,
    CHAR_SPACE,
        #if TEXT_MESSAGE_LENGTH == 7
    CHAR_SPACE
        #endif
    #endif
};

static const lcd_cell_t lcd_no_power[TEXT_MESSAGE_LENGTH] =
{
    CHAR_b,
    CHAR_l,
    CHAR_SPACE,
    CHAR_o,
    CHAR_u,
    CHAR_t,
    #if TEXT_MESSAGE_LENGTH == 7
    CHAR_SPACE
    #endif
};

static const lcd_cell_t lcd_4v2_power[TEXT_MESSAGE_LENGTH] =
{
    CHAR_SPACE,
    CHAR_SPACE,
    CHAR_SPACE,
    CHAR_4,
    #if defined(USE_STARBURST)
    CHAR_V,
    #else
    CHAR_U,
    #endif
    CHAR_2,
    #if TEXT_MESSAGE_LENGTH == 7
    CHAR_SPACE
    #endif
};

static const lcd_cell_t lcd_normal_power[TEXT_MESSAGE_LENGTH] =
{
    CHAR_SPACE,
    CHAR_SPACE,
    CHAR_SPACE,
    CHAR_8,
    #if defined(USE_STARBURST)
    CHAR_V,
    #else
    CHAR_U,
    #endif
    CHAR_4,
    #if TEXT_MESSAGE_LENGTH == 7
    CHAR_SPACE
    #endif
};

    #if !defined(SINGLE_PHASE)  &&  defined(ICON_PHASE_A)  &&  defined(ICON_PHASE_B)  &&  defined(ICON_PHASE_C)
static uint8_t phase_icons[NUM_PHASES] =
{
    ICON_PHASE_A,
    ICON_PHASE_B,
    ICON_PHASE_C
};
    #endif

    #if defined(__MSP430__)
enum
{
        #if defined(PER_PHASE_ENERGY_SUPPORT)
            #if !defined(TWO_LINE_LCD)
    DISPLAY_STAGE_PHASE_POWER,
            #endif
    DISPLAY_STAGE_PHASE_ENERGY,
        #endif
        #if defined(IRMS_SUPPORT)  &&  defined(VRMS_SUPPORT)  &&  defined(POWER_FACTOR_SUPPORT)
            #if defined(REACTIVE_POWER_SUPPORT)
    DISPLAY_STAGE_REACTIVE_POWER,
            #endif
            #if defined(APPARENT_POWER_SUPPORT)
    DISPLAY_STAGE_VA,
            #endif
    DISPLAY_STAGE_POWER_FACTOR,
        #endif
        #if defined(VRMS_SUPPORT)
    DISPLAY_STAGE_VOLTAGE,
        #endif
        #if defined(IRMS_SUPPORT)
    DISPLAY_STAGE_CURRENT,
        #endif
        #if defined(MAINS_FREQUENCY_SUPPORT)
    DISPLAY_STAGE_MAINS_FREQUENCY,
        #endif
        #if !defined(SINGLE_PHASE)
    DISPLAY_STAGE_PHASE_LAST,
        #endif
        #if !defined(SINGLE_PHASE)  &&  defined(NEUTRAL_MONITOR_SUPPORT)  &&  defined(IRMS_SUPPORT)
    DISPLAY_STAGE_NEUTRAL_CURRENT,
        #endif
        #if defined(TOTAL_ENERGY_SUPPORT)
    DISPLAY_STAGE_TOTAL_ACTIVE_POWER,
            #if !defined(TWO_LINE_LCD)
    DISPLAY_STAGE_TOTAL_ENERGY,
            #endif
        #endif
        #if defined(RTC_SUPPORT)
    DISPLAY_STAGE_DATE,
            #if !defined(DEDICATED_TIME_FIELD)  &&  !defined(TWO_LINE_LCD)
    DISPLAY_STAGE_TIME,
            #endif
        #endif
        #if defined(TEMPERATURE_SUPPORT)
    DISPLAY_STAGE_TEMPERATURE,
        #endif
        #if defined(MULTI_RATE_SUPPORT)
    DISPLAY_STAGE_CURRENT_TARIFF,
        #endif
    DISPLAY_STAGE_LAST
};

static int8_t display_stage;
        #if !defined(SINGLE_PHASE)
static int8_t display_phase;
        #endif

        #if !defined(USE_STARBURST)
#define LCDcharsx LCDchars
        #else
void LCDcharsx(const lcd_cell_t *s, int pos, int len)
{
    if (lcd_pos_step < 0)
        pos = -pos;
    if (abs(lcd_pos_step) > 1)
        pos <<= 1;
    pos = lcd_pos_base + pos;
    for (  ;  len > 0;  --len)
    {
        LCDMEM[pos] = *s & 0xFF;
        pos += (lcd_pos_step >> 1);
        LCDMEM[pos] = *s >> 8;
        s++;
        pos += (lcd_pos_step >> 1);
    }
}
        #endif

#if defined(FIRST_ROW_START)
void display_clear_line_1(void)
{
    int i;
    
    for (i = FIRST_ROW_START;  i < FIRST_ROW_START + FIRST_ROW_CHARS;  i++)
        LCDchar(CHAR_SPACE, i);
}
#endif

#if defined(SECOND_ROW_START)
void display_clear_line_2(void)
{
    int i;
    
    for (i = SECOND_ROW_START;  i < SECOND_ROW_START + SECOND_ROW_CHARS;  i++)
        LCDchar(CHAR_SPACE, i);
}
#endif

void display_power_fail_message(void)
{
    LCDchar(CHAR_SPACE, 1);
    LCDcharsx(lcd_no_power, FIRST_POSITION, TEXT_MESSAGE_LENGTH);
}

void display_startup_message(void)
{
    LCDchar(CHAR_SPACE, 1);
    LCDcharsx(lcd_startup, FIRST_POSITION, TEXT_MESSAGE_LENGTH);
}

void display_power_4v2_message(void)
{
    LCDchar(CHAR_SPACE, 1);
    LCDcharsx(lcd_4v2_power, FIRST_POSITION, TEXT_MESSAGE_LENGTH);
}

void display_power_normal_message(void)
{
    LCDchar(CHAR_SPACE, 1);
    LCDcharsx(lcd_normal_power, FIRST_POSITION, TEXT_MESSAGE_LENGTH);
}

static void LCDicon(int pos, int on)
{
    static const lcd_cell_t segs[] =
    {
        SEG_a,
        SEG_b,
        SEG_c,
        SEG_d,
        SEG_e,
        SEG_f,
        #if !defined(USE_STARBURST)
        SEG_g,
        SEG_h,
        #else
        SEG_i,
        SEG_h,
        SEG_12,
        SEG_1,
        SEG_3,
        SEG_5,
        SEG_6,
        SEG_7,
        SEG_9,
        SEG_11
        #endif
    };

    LCDmodify_char(segs[pos >> 5], pos & 0x1F, on);
}

static void LCDoverrange(void)
{
    LCDcharsx(lcd_high, FIRST_POSITION, TEXT_MESSAGE_LENGTH);
}

        #if defined(MAINS_FREQUENCY_SUPPORT)
            #if defined(SINGLE_PHASE)
static __inline__ void display_mains_frequency(void)
            #else
static __inline__ void display_mains_frequency(struct phase_parms_s *phase)
            #endif
{
    int32_t x;

    /* Display mains frequency in 0.1Hz or 0.01Hz increments */
            #if !defined(ICON_HERTZ)  &&  defined(DISPLAY_TYPE_POSITION)
    LCDchar(CHAR_F, DISPLAY_TYPE_POSITION);
            #endif

            #if defined(PRECALCULATED_PARAMETER_SUPPORT)
    x = phase->frequency;
            #else
                #if defined(SINGLE_PHASE)
    x = frequency();
                #else
    x = frequency(phase);
                #endif
            #endif
    LCDdecu32(x, FIRST_POSITION, NUMBER_WIDTH, FREQUENCY_RESOLUTION);
            #if defined(ICON_DECIMAL_FREQUENCY)
    LCDicon(ICON_DECIMAL_FREQUENCY, TRUE);
            #endif
            #if defined(ICON_HERTZ)
    LCDicon(ICON_HERTZ, TRUE);
            #endif
}
        #endif

        #if defined(VRMS_SUPPORT)
            #if defined(SINGLE_PHASE)
static void display_vrms(void)
            #else
static __inline__ void display_vrms(struct phase_parms_s *phase, struct phase_nv_parms_s const *phase_nv)
            #endif
{
    int32_t x;

    /* Display RMS voltage in 0.1V or 0.01V increments */
            #if defined(PRECALCULATED_PARAMETER_SUPPORT)
    if (phase->V_rms == 0xFFFF)
        x = -1;
    else
        x = phase->V_rms;
            #else
                #if defined(SINGLE_PHASE)
    x = voltage();
                #else
    x = voltage(phase, phase_nv);
                #endif
            #endif
            #if !defined(ICON_VOLTAGE)  &&  defined(DISPLAY_TYPE_POSITION)
                #if defined(USE_STARBURST)
    LCDchar(CHAR_V, DISPLAY_TYPE_POSITION);
                #else
    LCDchar(CHAR_U, DISPLAY_TYPE_POSITION);
                #endif
            #endif
    if (x < 0)
    {
        LCDoverrange();
    }
    else
    {
            #if defined(VOLTAGE_DISPLAY_DIVISOR)
        x /= VOLTAGE_DISPLAY_DIVISOR;
            #endif
        LCDdecu32(x, FIRST_POSITION, NUMBER_WIDTH, VOLTAGE_RESOLUTION);
            #if defined(ICON_DECIMAL_VOLTAGE)
        LCDicon(ICON_DECIMAL_VOLTAGE, TRUE);
            #endif
    }
            #if defined(ICON_V)
    LCDicon(ICON_V, TRUE);
            #endif
            #if defined(ICON_VOLTAGE)
    LCDicon(ICON_VOLTAGE, TRUE);
            #endif
}
        #endif

        #if defined(IRMS_SUPPORT)
            #if defined(SINGLE_PHASE)
static __inline__ void display_irms(void)
            #else
static __inline__ void display_irms(struct phase_parms_s *phase, struct phase_nv_parms_s const *phase_nv)
            #endif
{
    int32_t x;

    /* Display RMS current in 1mA increments */
            #if defined(PRECALCULATED_PARAMETER_SUPPORT)
    if (phase->I_rms == 0xFFFF)
        x = -1;
    else
        x = phase->I_rms;
            #else
                #if defined(SINGLE_PHASE)
    x = current();
                #else
    x = current(phase, phase_nv);
                #endif
            #endif
            #if !defined(ICON_CURRENT)  &&  defined(DISPLAY_TYPE_POSITION)
    LCDchar(CHAR_C, DISPLAY_TYPE_POSITION);
            #endif

    if (x < 0)
    {
        LCDoverrange();
    }
    else
    {
            #if defined(CURRENT_DISPLAY_DIVISOR)
        x /= CURRENT_DISPLAY_DIVISOR;
            #endif
        LCDdecu32(x, FIRST_POSITION, NUMBER_WIDTH, CURRENT_RESOLUTION);
            #if defined(ICON_DECIMAL_CURRENT)
        LCDicon(ICON_DECIMAL_CURRENT, TRUE);
            #endif
    }
            #if defined(ICON_A)
    LCDicon(ICON_A, TRUE);
            #endif
            #if defined(ICON_CURRENT)
    LCDicon(ICON_CURRENT, TRUE);
            #endif
}
        #endif

        #if !defined(SINGLE_PHASE)  &&  defined(NEUTRAL_MONITOR_SUPPORT)  &&  defined(IRMS_SUPPORT)
static __inline__ void display_neutral_irms(void)
{
    int32_t x;

    /* Display RMS current in 1mA increments */
            #if defined(PRECALCULATED_PARAMETER_SUPPORT)
    if (neutral.I_rms == 0xFFFF)
        x = -1;
    else
        x = neutral.I_rms;
            #else
    x = current(xxx);
            #endif
    if (x < 0)
    {
        LCDoverrange();
    }
    else
    {
            #if defined(CURRENT_DISPLAY_DIVISOR)
        x /= CURRENT_DISPLAY_DIVISOR;
            #endif
        LCDdecu32(x, FIRST_POSITION, NUMBER_WIDTH, CURRENT_RESOLUTION);
            #if defined(ICON_DECIMAL_CURRENT)
        LCDicon(ICON_DECIMAL_CURRENT, TRUE);
            #endif
    }
            #if !defined(ICON_CURRENT)  &&  defined(DISPLAY_TYPE_POSITION)
    LCDchar(CHAR_C, DISPLAY_TYPE_POSITION);
            #endif
            #if defined(ICON_A)
    LCDicon(ICON_A, TRUE);
            #endif
            #if defined(ICON_CURRENT)
    LCDicon(ICON_CURRENT, TRUE);
            #endif
}
        #endif

        #if defined(TOTAL_ENERGY_SUPPORT)
static __inline__ void display_total_consumed_energy(void)
{
    //Display energy in 0.1kWh increments
            #if !(defined(ICON_kW)  &&  defined(ICON_H))  &&  !defined(ICON_kWH)  &&  defined(DISPLAY_TYPE_POSITION)
    LCDchar(CHAR_E, DISPLAY_TYPE_POSITION);
            #endif
            #if FIRST_POSITION > 2
    LCDchar(CHAR_SPACE, 2);
            #endif

            #if ENERGY_RESOLUTION == 1
    LCDdecu32(total_consumed_energy/10L, FIRST_POSITION, NUMBER_WIDTH, ENERGY_RESOLUTION);
            #else
    LCDdecu32(total_consumed_energy/100L, FIRST_POSITION, NUMBER_WIDTH, ENERGY_RESOLUTION);
            #endif
            #if defined(ICON_DECIMAL_ENERGY)
    LCDicon(ICON_DECIMAL_ENERGY, TRUE);
            #endif
            #if defined(ICON_kWH)
    LCDicon(ICON_kWH, TRUE);
            #elif defined(ICON_kW)  &&  defined(ICON_H)
    LCDicon(ICON_kW, TRUE);
    LCDicon(ICON_H, TRUE);
            #endif
}
        #endif

        #if defined(PER_PHASE_ENERGY_SUPPORT)
            #if defined(SINGLE_PHASE)
static __inline__ void display_phase_consumed_energy(void)
            #else
static __inline__ void display_phase_consumed_energy(struct phase_parms_s *phase)
            #endif
{
    //Display energy in 0.1kWh increments
            #if !(defined(ICON_kW)  &&  defined(ICON_H))  &&  !defined(ICON_kWH)  &&  defined(DISPLAY_TYPE_POSITION)
    LCDchar(CHAR_E, DISPLAY_TYPE_POSITION);
            #endif
            #if ENERGY_RESOLUTION == 1
    LCDdecu32(phase->consumed_energy/10L, FIRST_ENERGY_POSITION, NUMBER_WIDTH, ENERGY_RESOLUTION);
            #else
    LCDdecu32(phase->consumed_energy/100L, FIRST_ENERGY_POSITION, NUMBER_WIDTH, ENERGY_RESOLUTION);
            #endif
            #if defined(ICON_DECIMAL_ENERGY)
    LCDicon(ICON_DECIMAL_ENERGY, TRUE);
            #endif
            #if defined(ICON_kWH)
    LCDicon(ICON_kWH, TRUE);
            #elif defined(ICON_kW)  &&  defined(ICON_H)
    LCDicon(ICON_kW, TRUE);
    LCDicon(ICON_H, TRUE);
            #endif
}
        #endif

        #if defined(TOTAL_ENERGY_SUPPORT)
static __inline__ void display_total_active_power(void)
{
    /* Display total power (all phased summed) in 0.01W increments */
            #if !defined(ICON_kW)  &&  defined(DISPLAY_TYPE_POSITION)
    LCDchar(CHAR_P, DISPLAY_TYPE_POSITION);
            #endif
            #if FIRST_ACTIVE_POWER_POSITION > 2
    LCDchar(CHAR_SPACE, 2);
            #endif
            #if POWER_RESOLUTION == 1
    LCDdecu32(total_active_power/10L, FIRST_ACTIVE_POWER_POSITION, NUMBER_WIDTH, POWER_RESOLUTION);
            #else
    LCDdecu32(total_active_power, FIRST_ACTIVE_POWER_POSITION, NUMBER_WIDTH, POWER_RESOLUTION);
            #endif
            #if defined(ICON_DECIMAL_POWER)
    LCDicon(ICON_DECIMAL_POWER, TRUE);
            #endif
            #if defined(ICON_kW)
    LCDicon(ICON_kW, TRUE);
            #endif
}
        #endif

        #if defined(PER_PHASE_ENERGY_SUPPORT)
            #if defined(SINGLE_PHASE)
static __inline__ void display_phase_power(void)
            #else
static __inline__ void display_phase_power(struct phase_parms_s *phase)
            #endif
{
    //Display per phase power in 0.01W increments
            #if !defined(ICON_kW)  &&  defined(DISPLAY_TYPE_POSITION)
    LCDchar(CHAR_P, DISPLAY_TYPE_POSITION);
            #endif
            #if POWER_RESOLUTION == 1
    LCDdecu32(phase->active_power/10L, FIRST_ACTIVE_POWER_POSITION, NUMBER_WIDTH, POWER_RESOLUTION);
            #else
    LCDdecu32(phase->active_power, FIRST_ACTIVE_POWER_POSITION, NUMBER_WIDTH, POWER_RESOLUTION);
            #endif
            #if defined(ICON_DECIMAL_POWER)
    LCDicon(ICON_DECIMAL_POWER, TRUE);
            #endif
            #if defined(ICON_kW)
    LCDicon(ICON_kW, TRUE);
            #endif
}
        #endif

        #if defined(IRMS_SUPPORT)  &&  defined(VRMS_SUPPORT)  &&  defined(POWER_FACTOR_SUPPORT)
            #if defined(SINGLE_PHASE)
static __inline__ void display_power_factor(void)
            #else
static __inline__ void display_power_factor(struct phase_parms_s *phase, struct phase_nv_parms_s const *phase_nv)
            #endif
{
    int16_t x;

            #if !defined(ICON_COS_PHI)  &&  defined(DISPLAY_TYPE_POSITION)
    LCDchar(CHAR_F, DISPLAY_TYPE_POSITION);
            #endif
            #if defined(PRECALCULATED_PARAMETER_SUPPORT)
    x = phase->power_factor;
            #else
                #if defined(SINGLE_PHASE)
    x = power_factor();
                #else
    x = power_factor(phase, phase_nv);
                #endif
            #endif
    if (x < 0)
    {
        LCDchar(CHAR_L, FIRST_POWER_FACTOR_POSITION);
        x = -x;
    }
    else
    {
        LCDchar(CHAR_C, FIRST_POWER_FACTOR_POSITION);
    }
    LCDdec16(x/10, FIRST_POWER_FACTOR_POSITION + 1, NUMBER_WIDTH - 2, 2);
            #if defined(ICON_DECIMAL_2)
    LCDicon(ICON_DECIMAL_2, TRUE);
            #endif
            #if defined(ICON_COS_PHI)
    LCDicon(ICON_COS_PHI, TRUE);
            #endif
}

            #if defined(REACTIVE_POWER_SUPPORT)
                #if defined(SINGLE_PHASE)
static __inline__ void display_reactive_power(void)
                #else
static __inline__ void display_reactive_power(struct phase_parms_s *phase)
                #endif
{
    int32_t x;

    // power_factor [ie cos(phase angle)] = (v.i)/sqrt(v.v * i.i)
    // real power = (2/N) * v.i
    // imag (reactive) power = (2/N) * sqrt(v.v * i.i - v.i * v.i)

    //Display reactive power in 0.01W increments
                #if !defined(ICON_kW)  &&  defined(DISPLAY_TYPE_POSITION)
                    #if defined(USE_STARBURST)
    LCDchar(CHAR_R, DISPLAY_TYPE_POSITION);
                    #else
    LCDchar(CHAR_r, DISPLAY_TYPE_POSITION);
                    #endif
                #endif
                #if defined(PRECALCULATED_PARAMETER_SUPPORT)
    x = labs(phase->reactive_power);
                #else
                    #if defined(SINGLE_PHASE)
    x = reactive_power();
                    #else
    x = reactive_power(phase);
                    #endif
                #endif
    //Needs scaling
                #if POWER_RESOLUTION == 1
    x /= 10;
                #endif
    LCDdecu32(x, FIRST_REACTIVE_POWER_POSITION, NUMBER_WIDTH, POWER_RESOLUTION);
    LCDicon(ICON_DECIMAL_POWER, TRUE);
                #if defined(ICON_kW)
    LCDicon(ICON_kW, TRUE);
                #endif
}
            #endif

            #if defined(APPARENT_POWER_SUPPORT)
                #if defined(SINGLE_PHASE)
static __inline__ void display_apparent_power(void)
                #else
static __inline__ void display_apparent_power(struct phase_parms_s *phase)
                #endif
{
    int32_t x;

    //Display apparent (VA) power in 0.01W increments
                #if !defined(ICON_kW)  &&  defined(DISPLAY_TYPE_POSITION)
                    #if defined(USE_STARBURST)
    LCDchar(CHAR_A, DISPLAY_TYPE_POSITION);
                    #else
    LCDchar(CHAR_n, DISPLAY_TYPE_POSITION);
                    #endif
                #endif
                #if defined(PRECALCULATED_PARAMETER_SUPPORT)
    x = phase->apparent_power;
                #else
                    #if defined(SINGLE_PHASE)
    x = apparent_power();
                    #else
    x = apparent_power(phase);
                    #endif
                #endif
    //Needs scaling
                #if POWER_RESOLUTION == 1
    x /= 10;
                #endif
    LCDdecu32(x, FIRST_APPARENT_POWER_POSITION, NUMBER_WIDTH, POWER_RESOLUTION);
    LCDicon(ICON_DECIMAL_POWER, TRUE);
                #if defined(ICON_kW)
    LCDicon(ICON_kW, TRUE);
                #endif
}
            #endif
        #endif

        #if defined(RTC_SUPPORT)
static /*__inline__*/ void display_current_date(void)
{
            #if !defined(ICON_DATE)  &&  defined(DISPLAY_TYPE_POSITION)
                #if defined(USE_STARBURST)
    LCDchar(CHAR_D, DISPLAY_TYPE_POSITION);
                #else
    LCDchar(CHAR_d, DISPLAY_TYPE_POSITION);
                #endif
                #if FIRST_POSITION > 2
    LCDchar(CHAR_t, DISPLAY_TYPE_POSITION + 1);
                #endif
            #endif
            #if defined(ZAP_COLON_CELL)
    LCDchar(CHAR_SPACE, ZAP_COLON_CELL);
            #endif
    LCDdecu16(rtc.year, YEAR_POSITION, 2, 1);
    LCDdecu16(rtc.month, MONTH_POSITION, 2, 1);
    LCDdecu16(rtc.day, DAY_POSITION, 2, 1);
            #if defined(ICON_DATE_COLON_1)
    LCDicon(ICON_DATE_COLON_1, TRUE);
            #endif
            #if defined(ICON_DATE_COLON_1A)
    LCDicon(ICON_DATE_COLON_1A, TRUE);
            #endif
            #if defined(ICON_DATE_COLON_2)
    LCDicon(ICON_DATE_COLON_2, TRUE);
            #endif
            #if defined(ICON_DATE_COLON_2A)
    LCDicon(ICON_DATE_COLON_2A, TRUE);
            #endif
            #if defined(ICON_DATE)
    LCDicon(ICON_DATE, TRUE);
            #endif
}

static __inline__ void display_current_time(void)
{
            #if defined(DEDICATED_TIME_FIELD)
    uint8_t x;

    /*TODO: This is fudged, as the time field has digits in
            reverse order from the main field. */
    LCDdecu16(rtc.hour, DEDICATED_TIME_FIELD + 2, 2, 1);
    LCDdecu16(rtc.minute, DEDICATED_TIME_FIELD, 2, 1);
    x = LCDMEM[12];
    LCDMEM[12] = LCDMEM[11];
    LCDMEM[11] = x;
    x = LCDMEM[14];
    LCDMEM[14] = LCDMEM[13];
    LCDMEM[13] = x;
                #if defined(ICON_TIME_FIELD_TIME)
    LCDicon(ICON_TIME_FIELD_TIME, TRUE);
                #endif
                #if defined(ICON_TIME_FIELD_COLON)
    LCDicon(ICON_TIME_FIELD_COLON, TRUE);
                #endif
            #elif defined(TWO_LINE_LCD)
                #if defined(ZAP_COLON_CELL)
    LCDchar(CHAR_SPACE, ZAP_COLON_CELL);
                #endif
    LCDdecu16(rtc.hour, HOUR_POSITION, 2, 1);
    LCDdecu16(rtc.minute, MINUTE_POSITION, 2, 1);
    LCDdecu16(rtc.second, SECONDS_POSITION, 2, 1);
                #if defined(ICON_TIME_COLON_1)
    LCDicon(ICON_TIME_COLON_1, TRUE);
                #endif
                #if defined(ICON_TIME_COLON_1A)
    LCDicon(ICON_TIME_COLON_1A, TRUE);
                #endif
                #if defined(ICON_TIME_COLON_2)
    LCDicon(ICON_TIME_COLON_2, TRUE);
                #endif
                #if defined(ICON_TIME_COLON_2A)
    LCDicon(ICON_TIME_COLON_2A, TRUE);
                #endif
                #if defined(ICON_TIME)
    LCDicon(ICON_TIME, TRUE);
                #endif
            #else
                #if !defined(ICON_TIME)  &&  defined(DISPLAY_TYPE_POSITION)
                    #if defined(USE_STARBURST)
    LCDchar(CHAR_T, DISPLAY_TYPE_POSITION);
                    #else
    LCDchar(CHAR_t, DISPLAY_TYPE_POSITION);
                    #endif
                    #if FIRST_POSITION > 2
    LCDchar(CHAR_i, DISPLAY_TYPE_POSITION + 1);
                    #endif
                #endif
                #if defined(ZAP_COLON_CELL)
    LCDchar(CHAR_SPACE, ZAP_COLON_CELL);
                #endif
    LCDdecu16(rtc.hour, HOUR_POSITION, 2, 1);
    LCDdecu16(rtc.minute, MINUTE_POSITION, 2, 1);
    LCDdecu16(rtc.second, SECONDS_POSITION, 2, 1);
                #if defined(ICON_TIME_COLON_1)
    LCDicon(ICON_TIME_COLON_1, TRUE);
                #endif
                #if defined(ICON_TIME_COLON_1A)
    LCDicon(ICON_TIME_COLON_1A, TRUE);
                #endif
                #if defined(ICON_TIME_COLON_2)
    LCDicon(ICON_TIME_COLON_2, TRUE);
                #endif
                #if defined(ICON_TIME_COLON_2A)
    LCDicon(ICON_TIME_COLON_2A, TRUE);
                #endif
                #if defined(ICON_TIME)
    LCDicon(ICON_TIME, TRUE);
                #endif
            #endif
}
        #endif

        #if defined(TEMPERATURE_SUPPORT)
static __inline__ void display_temperature(void)
{
    int32_t temp;

    LCDchar(CHAR_C, DISPLAY_TYPE_POSITION);
    /* Convert the temperature reading to degrees C */
    /* DegC = ((((int32_t) ADC_result - 1615)*704)/4095); */
    /* We filtered in a way that multiplied the temperature reading by 8 */
    /* Therefore, to get a result in 0.1 degree C steps we do this... */
    temp = temperature - nv_parms.seg_a.s.temperature_offset;
    temp *= nv_parms.seg_a.s.temperature_scaling;
    temp >>= 16;
    LCDdec32(temp, FIRST_TEMPERATURE_POSITION, NUMBER_WIDTH - 1, 1);
}
        #endif

        #if defined(MULTI_RATE_SUPPORT)
void display_current_tariff(void)
{
            #if !defined(ICON_DATE)  &&  defined(DISPLAY_TYPE_POSITION)
                #if defined(USE_STARBURST)
    LCDchar(CHAR_T, DISPLAY_TYPE_POSITION);
                #else
    LCDchar(CHAR_t, DISPLAY_TYPE_POSITION);
                #endif
    LCDchar(CHAR_a, DISPLAY_TYPE_POSITION + 1);
    LCDchar(CHAR_r, DISPLAY_TYPE_POSITION + 2);
    LCDchar(CHAR_r, DISPLAY_TYPE_POSITION + 3);
    LCDchar(CHAR_SPACE, DISPLAY_TYPE_POSITION + 4);
            #endif
    LCDdecu16(current_tariff + 1, DISPLAY_TYPE_POSITION + 5, 2, 0);
}

void display_tariff_holiday(void)
{
    int i;
    eeprom_holiday_t holiday;

    info_step = 0;
    for (i = info_step;  i < MULTIRATE_MAX_HOLIDAYS;  i++)
    {
        iicEEPROM_read(EEPROM_START_HOLIDAYS + i*sizeof(eeprom_holiday_t), (void *) &holiday, sizeof(holiday));
        if (holiday.year)
        {
            info_step = i;
            #if !defined(ICON_DATE)  &&  defined(DISPLAY_TYPE_POSITION)
                #if defined(USE_STARBURST)
            LCDchar(CHAR_D, DISPLAY_TYPE_POSITION);
                #else
            LCDchar(CHAR_d, DISPLAY_TYPE_POSITION);
                #endif
                #if FIRST_POSITION > 2
            LCDchar(CHAR_t, DISPLAY_TYPE_POSITION + 1);
                #endif
            #endif
            #if defined(ZAP_COLON_CELL)
            LCDchar(CHAR_SPACE, ZAP_COLON_CELL);
            #endif
            LCDdecu16(holiday.year, YEAR_POSITION, 2, 1);
            LCDdecu16(holiday.month, MONTH_POSITION, 2, 1);
            LCDdecu16(holiday.day, DAY_POSITION, 2, 1);
            #if defined(ICON_COLON_1)
            LCDicon(ICON_COLON_1, TRUE);
            #endif
            #if defined(ICON_COLON_1A)
            LCDicon(ICON_COLON_1A, TRUE);
            #endif
            #if defined(ICON_COLON_2)
            LCDicon(ICON_COLON_2, TRUE);
            #endif
            #if defined(ICON_COLON_2A)
            LCDicon(ICON_COLON_2A, TRUE);
            #endif
            #if defined(ICON_DATE)
            LCDicon(ICON_DATE, TRUE);
            #endif
            return;
        }
    }
}
        #endif

void update_display(void)
{
    int skip_stage;
        #if !defined(SINGLE_PHASE)
    struct phase_parms_s *phase;
    struct phase_nv_parms_s const *phase_nv;

    phase = &chan[(int) display_phase];
    phase_nv = &nv_parms.seg_a.s.chan[(int) display_phase];
        #endif
    do
    {
        skip_stage = FALSE;
        LCDchar(CHAR_SPACE, DISPLAY_TYPE_POSITION);
        #if defined(ICON_ONLY_1)
        LCDchar(0, ICON_ONLY_1);
        #endif
        #if defined(ICON_ONLY_2)
        LCDchar(0, ICON_ONLY_2);
        #endif
        #if defined(ICON_ONLY_3)
        LCDchar(0, ICON_ONLY_3);
        #endif
        #if defined(PHASE_POSITION)
            #if defined(SINGLE_PHASE)
        LCDchar(CHAR_SPACE, PHASE_POSITION);
            #else
        if (display_stage <= DISPLAY_STAGE_PHASE_LAST)
            LCDdecu16(display_phase + 1, PHASE_POSITION, 1, 0);
        else
            LCDchar(CHAR_SPACE, PHASE_POSITION);
            #endif
        #endif
        display_clear_line_1();
        #if defined(TWO_LINE_LCD)
        display_clear_line_2();
        #endif
        switch (display_stage)
        {
        #if defined(PER_PHASE_ENERGY_SUPPORT)
        case DISPLAY_STAGE_PHASE_ENERGY:
            #if defined(SINGLE_PHASE)
            display_phase_consumed_energy();
            #else
            display_phase_consumed_energy(phase);
            #endif
            #if !defined(TWO_LINE_LCD)
            break;
        case DISPLAY_STAGE_PHASE_POWER:
            #endif
            #if defined(SINGLE_PHASE)
            display_phase_power();
            #else
            display_phase_power(phase);
            #endif
            break;
        #endif
        #if defined(MAINS_FREQUENCY_SUPPORT)
        case DISPLAY_STAGE_MAINS_FREQUENCY:
            #if defined(SINGLE_PHASE)
            display_mains_frequency();
            #else
            display_mains_frequency(phase);
            #endif
            break;
        #endif
        #if defined(IRMS_SUPPORT)
        case DISPLAY_STAGE_CURRENT:
            #if defined(SINGLE_PHASE)
            display_irms();
            #else
            display_irms(phase, phase_nv);
            #endif
            break;
        #endif
        #if defined(VRMS_SUPPORT)
        case DISPLAY_STAGE_VOLTAGE:
            #if defined(SINGLE_PHASE)
            display_vrms();
            #else
            display_vrms(phase, phase_nv);
            #endif
            break;
        #endif
        #if defined(IRMS_SUPPORT)  &&  defined(VRMS_SUPPORT)  &&  defined(POWER_FACTOR_SUPPORT)
        case DISPLAY_STAGE_POWER_FACTOR:
            #if defined(SINGLE_PHASE)
            display_power_factor();
            #else
            display_power_factor(phase, phase_nv);
            #endif
            break;
            #if defined(REACTIVE_POWER_SUPPORT)
        case DISPLAY_STAGE_REACTIVE_POWER:
                #if defined(SINGLE_PHASE)
            display_reactive_power();
                #else
            display_reactive_power(phase);
                #endif
            break;
            #endif
            #if defined(APPARENT_POWER_SUPPORT)
        case DISPLAY_STAGE_VA:
                #if defined(SINGLE_PHASE)
            display_apparent_power();
                #else
            display_apparent_power(phase);
                #endif
            break;
            #endif
        #endif
    #endif
    #if defined(RTC_SUPPORT)
        case DISPLAY_STAGE_DATE:
            display_current_date();
        #if defined(TWO_LINE_LCD)
            display_current_time();
        #endif
            break;
        #if !defined(DEDICATED_TIME_FIELD)  &&  !defined(TWO_LINE_LCD)
        case DISPLAY_STAGE_TIME:
            display_current_time();
            break;
        #endif
    #endif
    #if !defined(SINGLE_PHASE)  &&  defined(NEUTRAL_MONITOR_SUPPORT)
        case DISPLAY_STAGE_NEUTRAL_CURRENT:
            display_neutral_irms();
            break;
    #endif
    #if defined(TEMPERATURE_SUPPORT)
        case DISPLAY_STAGE_TEMPERATURE:
            display_temperature();
            break;
    #endif
    #if defined(MULTI_RATE_SUPPORT)
        case DISPLAY_STAGE_CURRENT_TARIFF:
            display_current_tariff();
            break;
    #endif
    #if defined(TOTAL_ENERGY_SUPPORT)
        case DISPLAY_STAGE_TOTAL_ACTIVE_POWER:
            display_total_active_power();
        #if !defined(TWO_LINE_LCD)
            break;
        default:
        #endif
            display_total_consumed_energy();
            break;
    #endif
        }

    #if !defined(SINGLE_PHASE)  &&  defined(ICON_PHASE_A)  &&  defined(ICON_PHASE_B)  &&  defined(ICON_PHASE_C)
        if (display_stage <= DISPLAY_STAGE_PHASE_LAST)
        {
            //Turn on the appropriate cursor, to indicate the current phase
            LCDicon(phase_icons[0], display_phase == 0);
            LCDicon(phase_icons[1], display_phase == 1);
            LCDicon(phase_icons[2], display_phase == 2);
        }
        else
        {
            //Turn off the cursors, as we are not showing phase related info
            LCDicon(phase_icons[0], FALSE);
            LCDicon(phase_icons[1], FALSE);
            LCDicon(phase_icons[2], FALSE);
        }
        #if FIRST_POSITION > 2
        LCDchar(CHAR_SPACE, 2);
        #endif
    #endif

    #if defined(SINGLE_PHASE)
        if (++display_stage >= DISPLAY_STAGE_LAST) 
            display_stage = 0;
    #else
        if (display_phase < NUM_PHASES - 1)
        {
            if (++display_stage >= DISPLAY_STAGE_PHASE_LAST)
            {
                display_stage = 0;
                display_phase++;
            }
        }
        else
        {
            if (++display_stage == DISPLAY_STAGE_PHASE_LAST)
                display_stage++;
            if (display_stage >= DISPLAY_STAGE_LAST) 
            {
                display_stage = 0;
                display_phase = 0;
            }
        }
    #endif
    }
    while (skip_stage);
    #if defined(DEDICATED_TIME_FIELD)
    display_current_time();
    #endif
    #if defined(ICON_BATTERY)
    LCDicon(ICON_BATTERY, TRUE);
    #endif
    #if defined(LIMP_MODE_SUPPORT)
    if (operating_mode == OPERATING_MODE_LIMP)
        LCDchar(CHAR_L, DISPLAY_TYPE_POSITION);
    #endif
}

#if !defined(__GNUC__)
int16_t ram_exclusion_zone_start_;
int16_t ram_exclusion_zone_middle_;
int16_t ram_exclusion_zone_end_;
#endif

#endif

#if defined(__MSP430__)
    #if defined(BASIC_KEYPAD_SUPPORT)
void keypad_handler(void)
{
    if ((key_states & KEY_1_DOWN))
    {
        update_display();
        key_states &= ~KEY_1_DOWN;
    }
    if ((key_states & KEY_1_REPEAT_DOWN))
    {
        update_display();
        key_states &= ~KEY_1_REPEAT_DOWN;
    }
}
    #endif
#endif

#if 0
    switch (info_section)
    {
    case 0:
        info_substep
        if (++info_step >= MULTIRATE_TARIFFS)
        {
            info_section++;
            info_step = 0;
        }
        break;
    case 1:
        if (++info_step >= MULTIRATE_MAX_HOLIDAYS)
        {
            info_section++;
            info_step = 0;
        }
        break;
    case 2:
        if (++info_step >= MULTIRATE_MAX_CUTOFF_DATES)
        {
            info_section++;
            info_step = 0;
        }
        break;
    case 3:
        if (++info_step >= MULTIRATE_HISTORIES)
        {
            info_section++;
            info_step = 0;
        }
        break;
    case 4:
        if (++info_step >= MULTIRATE_MAX_DAILY_PEAKS)
        {
            info_section++;
            info_step = 0;
        }
        break;
    }
#endif
